<?php

namespace App\Services;

use App\Models\Order;
use App\Models\OrderItem;
use App\Models\Product;
use Maatwebsite\Excel\Facades\Excel;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use App\Services\InventoryService;

class ImportService
{
    protected $inventoryService;
    private $currentHeaders = null;
    
    public function __construct(InventoryService $inventoryService)
    {
        $this->inventoryService = $inventoryService;
    }
    
    /**
     * Import orders from Excel file
     */
    public function importOrdersFromExcel($file, bool $deductInventory = true): array
    {
        $results = [
            'success' => 0,
            'errors' => [],
            'skipped' => 0,
            'orders_created' => 0,
            'line_items_created' => 0,
            'inventory_warnings' => []
        ];
        
        try {
            $data = Excel::toArray([], $file);
            
            if (empty($data) || empty($data[0])) {
                throw new \Exception('Excel file is empty or invalid');
            }
            
            $rows = $data[0];
            $headers = array_shift($rows); // Remove header row
            
            // Validate required columns exist
            $this->validateOrderHeaders($headers);
            
            // Store headers for later use
            $this->currentHeaders = $headers;
            
            // Group rows by order
            $groupedOrders = $this->groupRowsByOrder($rows, $headers);
            
            DB::transaction(function () use ($groupedOrders, $deductInventory, &$results) {
                foreach ($groupedOrders as $orderNumber => $orderRows) {
                    try {
                        $order = $this->processCompleteOrder($orderRows, $orderNumber);
                        $results['success']++;
                        $results['orders_created']++;
                        $results['line_items_created'] += count($orderRows);
                        
                        // Check and deduct inventory if enabled
                        if ($deductInventory) {
                            $this->handleInventoryDeduction($order, $results);
                        }
                    } catch (\Exception $e) {
                        $results['errors'][] = "Order {$orderNumber}: " . $e->getMessage();
                        $results['skipped']++;
                    }
                }
            });
            
        } catch (\Exception $e) {
            $results['errors'][] = "File processing error: " . $e->getMessage();
        }
        
        return $results;
    }
    
    /**
     * Validate that required columns exist in the Excel file
     */
    private function validateOrderHeaders(array $headers): void
    {
        $requiredColumns = [
            'Name',
            'Lineitem name',
            'Lineitem quantity'
        ];
        
        foreach ($requiredColumns as $required) {
            $found = false;
            foreach ($headers as $header) {
                if (strcasecmp(trim($header), $required) === 0) {
                    $found = true;
                    break;
                }
            }
            
            if (!$found) {
                throw new \Exception("Required column '{$required}' not found in Excel file");
            }
        }
    }
    
    /**
     * Group rows by order number (Name column)
     */
    private function groupRowsByOrder(array $rows, array $headers): array
    {
        $nameIndex = $this->findColumnIndex($headers, ['Name', 'name', 'Order Number']);
        
        if ($nameIndex === false) {
            throw new \Exception('Order Name/Number column not found');
        }
        
        $grouped = [];
        $currentOrderName = null;
        
        foreach ($rows as $row) {
            // Get order name from current row
            $orderName = trim($row[$nameIndex] ?? '');
            
            // If order name is empty, use the last known order name (for multi-line items)
            if (empty($orderName)) {
                $orderName = $currentOrderName;
            } else {
                $currentOrderName = $orderName;
            }
            
            if (empty($orderName)) {
                continue; // Skip rows without order identification
            }
            
            if (!isset($grouped[$orderName])) {
                $grouped[$orderName] = [];
            }
            
            $grouped[$orderName][] = $row;
        }
        
        return $grouped;
    }
    
    /**
     * Process a complete order with all its line items
     */
    private function processCompleteOrder(array $orderRows, string $orderNumber): Order
    {
        // Use the first row to get order-level data
        $firstRow = $orderRows[0];
        
        // Extract column indices
        $headers = $this->getExpectedHeaders();
        $columnIndices = $this->mapHeaderIndices($headers);
        
        // Build order data from first row
        $orderData = [
            'shopify_order_id' => $orderNumber,
            'customer_name' => $this->getValueFromRow($firstRow, $columnIndices, ['Billing Name', 'Name'], 'Unknown Customer'),
            'customer_email' => $this->getValueFromRow($firstRow, $columnIndices, ['Email'], null),
            'total_amount' => $this->parseAmount($this->getValueFromRow($firstRow, $columnIndices, ['Total'], '0')),
            'status' => $this->mapOrderStatus($this->getValueFromRow($firstRow, $columnIndices, ['Financial Status', 'Fulfillment Status'], 'pending')),
            'order_date' => now()->format('Y-m-d'),
            'notes' => null
        ];
        
        // Check if order already exists
        if (Order::where('shopify_order_id', $orderNumber)->exists()) {
            throw new \Exception("Order already exists in the system");
        }
        
        // Create order
        $order = Order::create($orderData);
        
        // Process each line item
        foreach ($orderRows as $row) {
            $this->processLineItem($order, $row, $columnIndices);
        }
        
        return $order;
    }
    
    /**
     * Handle inventory deduction for an order
     */
    private function handleInventoryDeduction(Order $order, array &$results): void
    {
        try {
            // Check stock availability first
            $shortages = $this->inventoryService->checkStockAvailability($order);
            
            if (!empty($shortages)) {
                $warningMessage = "Order {$order->shopify_order_id} has inventory shortages: ";
                foreach ($shortages as $shortage) {
                    $warningMessage .= "{$shortage['raw_material']->name} (need {$shortage['required']}, have {$shortage['available']}); ";
                }
                $results['inventory_warnings'][] = $warningMessage;
            }
            
            // Deduct inventory anyway (will result in negative stock if insufficient)
            $this->inventoryService->updateRawMaterialStock($order);
            
            // Update order status to indicate it's been processed
            $order->update(['status' => 'processing']);
            
        } catch (\Exception $e) {
            $results['inventory_warnings'][] = "Order {$order->shopify_order_id}: Inventory deduction failed - {$e->getMessage()}";
        }
    }
    
    /**
     * Get expected headers based on Shopify format
     */
    private function getExpectedHeaders(): array
    {
        // This will be set dynamically from the actual file
        return $this->currentHeaders ?? [];
    }
    
    /**
     * Map header indices for quick lookup
     */
    private function mapHeaderIndices(array $headers): array
    {
        $indices = [];
        foreach ($headers as $index => $header) {
            $indices[trim($header)] = $index;
        }
        return $indices;
    }
    
    /**
     * Get value from row using multiple possible column names
     */
    private function getValueFromRow(array $row, array $columnIndices, array $possibleNames, $default = null)
    {
        foreach ($possibleNames as $name) {
            if (isset($columnIndices[$name]) && isset($row[$columnIndices[$name]])) {
                $value = trim($row[$columnIndices[$name]]);
                if (!empty($value)) {
                    return $value;
                }
            }
        }
        return $default;
    }
    
    /**
     * Process a single line item
     */
    private function processLineItem(Order $order, array $row, array $columnIndices): void
    {
        $lineItemName = $this->getValueFromRow($row, $columnIndices, ['Lineitem name'], null);
        $lineItemQuantity = $this->getValueFromRow($row, $columnIndices, ['Lineitem quantity'], '0');
        $lineItemPrice = $this->getValueFromRow($row, $columnIndices, ['Lineitem price'], '0');
        $lineItemSku = $this->getValueFromRow($row, $columnIndices, ['Lineitem sku'], null);
        
        if (empty($lineItemName)) {
            throw new \Exception("Line item name is required");
        }
        
        $quantity = (int) $lineItemQuantity;
        $price = $this->parseAmount($lineItemPrice);
        
        if ($quantity <= 0) {
            throw new \Exception("Line item quantity must be greater than 0");
        }
        
        // Find or create product
        $product = $this->findOrCreateProduct($lineItemName, $lineItemSku, $price);
        
        // Create order item
        OrderItem::create([
            'order_id' => $order->id,
            'product_id' => $product->id,
            'quantity' => $quantity,
            'unit_price' => $price,
            'total_price' => $price * $quantity
        ]);
    }
    
    /**
     * Find or create a product based on line item data
     */
    private function findOrCreateProduct(string $name, ?string $sku, float $price): Product
    {
        $query = Product::query();
        
        // Try to find by SKU first, then by name
        if (!empty($sku)) {
            $product = $query->where('sku', $sku)->first();
            if ($product) {
                // Update price if changed
                if ($product->price != $price) {
                    $product->update(['price' => $price]);
                }
                return $product;
            }
        }
        
        // Try to find by exact name match
        $product = Product::where('name', $name)->first();
        if ($product) {
            // Update price and SKU if provided
            $updateData = ['price' => $price];
            if (!empty($sku) && empty($product->sku)) {
                $updateData['sku'] = $sku;
            }
            $product->update($updateData);
            return $product;
        }
        
        // Create new product
        return Product::create([
            'name' => $name,
            'sku' => $sku ?? $this->generateSkuFromName($name),
            'price' => $price,
            'is_active' => true
        ]);
    }
    
    /**
     * Parse amount string to float
     */
    private function parseAmount(string $amount): float
    {
        // Remove currency symbols, commas, and spaces
        $cleaned = preg_replace('/[^0-9.]/', '', $amount);
        return (float) $cleaned;
    }
    
    /**
     * Process a single order row from Excel (DEPRECATED - kept for backward compatibility)
     */
    private function processOrderRow(array $row, array $headers, int $rowNumber): void
    {
        // Map Excel columns to our data structure
        $orderData = $this->mapOrderData($row, $headers);
        
        // Check if order already exists
        if (Order::where('shopify_order_id', $orderData['shopify_order_id'])->exists()) {
            throw new \Exception("Order {$orderData['shopify_order_id']} already exists");
        }
        
        // Create order
        $order = Order::create($orderData);
        
        // Process order items
        $this->processOrderItems($order, $row, $headers);
    }
    
    /**
     * Map Excel row data to order structure
     */
    private function mapOrderData(array $row, array $headers): array
    {
        $mappedData = [];
        
        // Common Shopify export columns mapping
        $columnMap = [
            'Name' => 'customer_name',
            'Email' => 'customer_email',
            'Total' => 'total_amount',
            'Created at' => 'order_date',
            'Order Number' => 'shopify_order_id',
            'Financial Status' => 'status',
            'Note' => 'notes'
        ];
        
        foreach ($headers as $index => $header) {
            $value = $row[$index] ?? null;
            
            if (isset($columnMap[$header]) && $value !== null) {
                $field = $columnMap[$header];
                
                switch ($field) {
                    case 'total_amount':
                        $mappedData[$field] = (float) str_replace(['$', ','], '', $value);
                        break;
                    case 'order_date':
                        $mappedData[$field] = \Carbon\Carbon::parse($value)->format('Y-m-d');
                        break;
                    case 'status':
                        $mappedData[$field] = $this->mapOrderStatus($value);
                        break;
                    default:
                        $mappedData[$field] = $value;
                }
            }
        }
        
        // Set defaults for required fields
        $mappedData['shopify_order_id'] = $mappedData['shopify_order_id'] ?? 'ORDER-' . time() . '-' . rand(1000, 9999);
        $mappedData['customer_name'] = $mappedData['customer_name'] ?? 'Unknown Customer';
        $mappedData['total_amount'] = $mappedData['total_amount'] ?? 0;
        $mappedData['order_date'] = $mappedData['order_date'] ?? now()->format('Y-m-d');
        $mappedData['status'] = $mappedData['status'] ?? 'pending';
        
        return $mappedData;
    }
    
    /**
     * Map Shopify status to our status
     */
    private function mapOrderStatus(string $shopifyStatus): string
    {
        $statusMap = [
            'paid' => 'processing',
            'pending' => 'pending',
            'partially_paid' => 'pending',
            'refunded' => 'cancelled',
            'voided' => 'cancelled',
            'partially_refunded' => 'processing'
        ];
        
        return $statusMap[$shopifyStatus] ?? 'pending';
    }
    
    /**
     * Process order items from Excel row
     */
    private function processOrderItems(Order $order, array $row, array $headers): void
    {
        // This is a simplified version - in reality, you'd need to parse
        // the line items from the Excel file based on your Shopify export format
        
        // For now, we'll create a placeholder item
        // You'll need to modify this based on your actual Excel structure
        $lineItems = $this->parseLineItems($row, $headers);
        
        foreach ($lineItems as $item) {
            // Find product by SKU or name
            $product = Product::where('sku', $item['sku'])
                ->orWhere('name', $item['name'])
                ->first();
            
            if (!$product) {
                // Create a new product if it doesn't exist
                $product = Product::create([
                    'name' => $item['name'],
                    'sku' => $item['sku'],
                    'price' => $item['price'],
                    'current_stock' => 0,
                    'min_stock_level' => 0,
                    'is_active' => true
                ]);
            }
            
            OrderItem::create([
                'order_id' => $order->id,
                'product_id' => $product->id,
                'quantity' => $item['quantity'],
                'unit_price' => $item['price'],
                'total_price' => $item['price'] * $item['quantity']
            ]);
        }
    }
    
    /**
     * Parse line items from Excel row
     * This is a placeholder - modify based on your Excel structure
     */
    private function parseLineItems(array $row, array $headers): array
    {
        // This is a simplified implementation
        // You'll need to modify this based on how your Shopify export structures line items
        
        $items = [];
        
        // Look for line item columns (this is just an example)
        foreach ($headers as $index => $header) {
            if (strpos($header, 'Lineitem') !== false) {
                $value = $row[$index] ?? null;
                if ($value) {
                    // Parse the line item data
                    // This is just a placeholder - you'll need to implement proper parsing
                    $items[] = [
                        'name' => $value,
                        'sku' => 'SKU-' . rand(1000, 9999),
                        'quantity' => 1,
                        'price' => 0
                    ];
                }
            }
        }
        
        // If no line items found, create a default one
        if (empty($items)) {
            $items[] = [
                'name' => 'Unknown Product',
                'sku' => 'UNKNOWN-' . time(),
                'quantity' => 1,
                'price' => 0
            ];
        }
        
        return $items;
    }
    
    /**
     * Import products from Excel file
     */
    public function importProductsFromExcel($file): array
    {
        $results = [
            'success' => 0,
            'errors' => [],
            'skipped' => 0
        ];
        
        try {
            $data = Excel::toArray([], $file);
            
            if (empty($data) || empty($data[0])) {
                throw new \Exception('Excel file is empty or invalid');
            }
            
            $rows = $data[0];
            $headers = array_shift($rows); // Remove header row
            
            // Validate headers
            $nameIndex = $this->findColumnIndex($headers, ['Name', 'name', 'Product Name', 'Product']);
            $priceIndex = $this->findColumnIndex($headers, ['Price', 'price', 'Unit Price']);
            
            if ($nameIndex === false) {
                throw new \Exception('Required column "Name" not found in Excel file');
            }
            
            if ($priceIndex === false) {
                throw new \Exception('Required column "Price" not found in Excel file');
            }
            
            DB::transaction(function () use ($rows, $nameIndex, $priceIndex, &$results) {
                foreach ($rows as $index => $row) {
                    try {
                        $this->processProductRow($row, $nameIndex, $priceIndex, $index + 2);
                        $results['success']++;
                    } catch (\Exception $e) {
                        $results['errors'][] = "Row " . ($index + 2) . ": " . $e->getMessage();
                        $results['skipped']++;
                    }
                }
            });
            
        } catch (\Exception $e) {
            $results['errors'][] = "File processing error: " . $e->getMessage();
        }
        
        return $results;
    }
    
    /**
     * Process a single product row from Excel
     */
    private function processProductRow(array $row, int $nameIndex, int $priceIndex, int $rowNumber): void
    {
        $name = trim($row[$nameIndex] ?? '');
        $price = $row[$priceIndex] ?? null;
        
        // Validate required fields
        if (empty($name)) {
            throw new \Exception("Product name is required");
        }
        
        // Clean and validate price
        if ($price === null || $price === '') {
            throw new \Exception("Price is required");
        }
        
        // Remove currency symbols and formatting
        $price = preg_replace('/[^0-9.]/', '', $price);
        $price = (float) $price;
        
        if ($price < 0) {
            throw new \Exception("Price cannot be negative");
        }
        
        // Generate SKU from name if product doesn't exist
        $sku = $this->generateSkuFromName($name);
        
        // Check if product already exists by name
        $existingProduct = Product::where('name', $name)->first();
        
        if ($existingProduct) {
            // Update existing product
            $existingProduct->update([
                'price' => $price,
            ]);
        } else {
            // Create new product
            Product::create([
                'name' => $name,
                'sku' => $sku,
                'price' => $price,
                'is_active' => true,
            ]);
        }
    }
    
    /**
     * Generate a unique SKU from product name
     */
    private function generateSkuFromName(string $name): string
    {
        // Convert to uppercase, replace spaces with dashes, remove special chars
        $baseSku = strtoupper(preg_replace('/[^A-Za-z0-9]+/', '-', $name));
        $baseSku = trim($baseSku, '-');
        
        // Limit length
        if (strlen($baseSku) > 20) {
            $baseSku = substr($baseSku, 0, 20);
        }
        
        // Check if SKU exists and add number suffix if needed
        $sku = $baseSku;
        $counter = 1;
        
        while (Product::where('sku', $sku)->exists()) {
            $sku = $baseSku . '-' . $counter;
            $counter++;
        }
        
        return $sku;
    }
    
    /**
     * Find column index by possible header names
     */
    private function findColumnIndex(array $headers, array $possibleNames)
    {
        foreach ($headers as $index => $header) {
            $header = trim($header);
            foreach ($possibleNames as $name) {
                if (strcasecmp($header, $name) === 0) {
                    return $index;
                }
            }
        }
        return false;
    }
    
    /**
     * Find courier tax column index (handles dynamic percentages like "WH Inc.Tax (2%)")
     */
    private function findCourierTaxColumnIndex(array $headers, string $baseColumnName): int|false
    {
        foreach ($headers as $index => $header) {
            $headerTrimmed = trim($header);
            
            // Match patterns like "WH Inc.Tax (2%)" or "WH SalesTax (3%)"
            if ($baseColumnName === 'WH Inc.Tax') {
                if (preg_match('/^WH\s+Inc\.?Tax\s*\(/i', $headerTrimmed)) {
                    return $index;
                }
            } elseif ($baseColumnName === 'WH SalesTax') {
                if (preg_match('/^WH\s+SalesTax\s*\(/i', $headerTrimmed)) {
                    return $index;
                }
            }
        }
        
        return false;
    }
    
    /**
     * Import courier data from Excel file
     */
    public function importCourierDataFromExcel($file): array
    {
        $results = [
            'success' => 0,
            'errors' => [],
            'skipped' => 0,
            'updated_orders' => []
        ];
        
        try {
            $data = Excel::toArray([], $file);
            
            if (empty($data) || empty($data[0])) {
                throw new \Exception('Excel file is empty or invalid');
            }
            
            $rows = $data[0];
            $headers = array_shift($rows); // Remove header row
            
            // Validate required columns exist
            $this->validateCourierHeaders($headers);
            
            // Map header indices
            $orderRefIndex = $this->findColumnIndex($headers, ['Order Ref.']);
            $shippingChargesIndex = $this->findColumnIndex($headers, ['ShippingCharges', 'Shipping Charges', 'Shipping']);
            $gstIndex = $this->findColumnIndex($headers, ['GST', 'Gst', 'gst']);
            $whIncTaxIndex = $this->findCourierTaxColumnIndex($headers, 'WH Inc.Tax');
            $whSalesTaxIndex = $this->findCourierTaxColumnIndex($headers, 'WH SalesTax');
            $netAmountIndex = $this->findColumnIndex($headers, ['NetAmount', 'Net Amount', 'Net']);
            
            DB::transaction(function () use ($rows, $orderRefIndex, $shippingChargesIndex, $gstIndex, $whIncTaxIndex, $whSalesTaxIndex, $netAmountIndex, &$results) {
                foreach ($rows as $index => $row) {
                    try {
                        $this->processCourierRow(
                            $row, 
                            $orderRefIndex, 
                            $shippingChargesIndex, 
                            $gstIndex, 
                            $whIncTaxIndex, 
                            $whSalesTaxIndex, 
                            $netAmountIndex, 
                            $index + 2,
                            $results
                        );
                        
                        $results['success']++;
                    } catch (\Exception $e) {
                        $results['errors'][] = "Row " . ($index + 2) . ": " . $e->getMessage();
                        $results['skipped']++;
                    }
                }
            });
            
        } catch (\Exception $e) {
            $results['errors'][] = "File processing error: " . $e->getMessage();
        }
        
        return $results;
    }
    
    /**
     * Validate that required columns exist in the courier Excel file
     */
    private function validateCourierHeaders(array $headers): void
    {
        $requiredColumns = [
            'Order Ref.' => false,
            'ShippingCharges' => false,
            'GST' => false,
            'WH Inc.Tax' => false,
            'WH SalesTax' => false,
            'NetAmount' => false
        ];
        
        foreach ($headers as $header) {
            $headerTrimmed = trim($header);
            
            // Check for Order Ref. (exact match with dot)
            if (strcasecmp($headerTrimmed, 'Order Ref.') === 0) {
                $requiredColumns['Order Ref.'] = true;
            }
            // Check for ShippingCharges
            elseif (stripos($headerTrimmed, 'ShippingCharges') !== false || stripos($headerTrimmed, 'Shipping Charges') !== false) {
                $requiredColumns['ShippingCharges'] = true;
            }
            // Check for GST
            elseif (strcasecmp($headerTrimmed, 'GST') === 0) {
                $requiredColumns['GST'] = true;
            }
            // Check for WH Inc.Tax with any percentage
            elseif (preg_match('/^WH\s+Inc\.?Tax\s*\(/i', $headerTrimmed)) {
                $requiredColumns['WH Inc.Tax'] = true;
            }
            // Check for WH SalesTax with any percentage
            elseif (preg_match('/^WH\s+SalesTax\s*\(/i', $headerTrimmed)) {
                $requiredColumns['WH SalesTax'] = true;
            }
            // Check for NetAmount
            elseif (stripos($headerTrimmed, 'NetAmount') !== false || stripos($headerTrimmed, 'Net Amount') !== false) {
                $requiredColumns['NetAmount'] = true;
            }
        }
        
        // Check if all required columns were found
        foreach ($requiredColumns as $column => $found) {
            if (!$found) {
                throw new \Exception("Required column '{$column}' not found in Excel file");
            }
        }
    }
    
    /**
     * Process a single courier data row from Excel
     */
    private function processCourierRow(
        array $row, 
        int $orderRefIndex, 
        int $shippingChargesIndex, 
        int $gstIndex, 
        int $whIncTaxIndex, 
        int $whSalesTaxIndex, 
        int $netAmountIndex, 
        int $rowNumber,
        array &$results
    ): void
    {
        $orderRef = trim($row[$orderRefIndex] ?? '');
        
        // Skip rows without Order Ref (empty rows or total rows)
        if (empty($orderRef)) {
            return;
        }
        
        // Find order by shopify_order_id
        $order = Order::where('shopify_order_id', $orderRef)->first();
        
        if (!$order) {
            throw new \Exception("Order '{$orderRef}' not found in the system");
        }
        
        // Parse amounts
        $shippingCharges = $this->parseAmount($row[$shippingChargesIndex] ?? '0');
        $gst = $this->parseAmount($row[$gstIndex] ?? '0');
        $whIncTax = $this->parseAmount($row[$whIncTaxIndex] ?? '0');
        $whSalesTax = $this->parseAmount($row[$whSalesTaxIndex] ?? '0');
        $netAmount = $this->parseAmount($row[$netAmountIndex] ?? '0');
        
        // Calculate total with shipping
        $totalWithShipping = $order->total_amount + $shippingCharges;
        
        // Update order with courier data
        $order->update([
            'shipping_charges' => $shippingCharges,
            'gst' => $gst,
            'wh_inc_tax' => $whIncTax,
            'wh_sales_tax' => $whSalesTax,
            'net_amount' => $netAmount,
            'courier_data_updated_at' => now()
        ]);
        
        $results['updated_orders'][] = [
            'order_ref' => $orderRef,
            'original_amount' => $order->total_amount,
            'shipping_charges' => $shippingCharges,
            'total_with_shipping' => $totalWithShipping,
            'deductions' => $gst + $whIncTax + $whSalesTax,
            'net_amount' => $netAmount
        ];
    }
    
    /**
     * Import returns from Excel file
     */
    public function importReturnsFromExcel($file, bool $restoreInventory = true): array
    {
        $results = [
            'success' => 0,
            'errors' => [],
            'skipped' => 0,
            'orders_marked_returned' => 0,
            'returned_products' => [], // Store product info with quantities
            'inventory_restored' => []
        ];
        
        try {
            $data = Excel::toArray([], $file);
            
            if (empty($data) || empty($data[0])) {
                throw new \Exception('Excel file is empty or invalid');
            }
            
            $rows = $data[0];
            $headers = array_shift($rows); // Remove header row
            
            // Validate required columns exist
            $this->validateReturnsHeaders($headers);
            
            // Find ORDER_REFERENCE_NUMBER column index
            $orderRefIndex = $this->findColumnIndex($headers, ['ORDER_REFERENCE_NUMBER']);
            
            DB::transaction(function () use ($rows, $orderRefIndex, $restoreInventory, &$results) {
                foreach ($rows as $index => $row) {
                    try {
                        // Skip empty rows
                        if (empty(array_filter($row))) {
                            continue;
                        }
                        
                        $orderRef = trim($row[$orderRefIndex] ?? '');
                        
                        if (empty($orderRef)) {
                            $results['skipped']++;
                            continue;
                        }
                        
                        // Find order by shopify_order_id
                        $order = Order::where('shopify_order_id', $orderRef)
                            ->with('orderItems.product')
                            ->first();
                        
                        if (!$order) {
                            $results['errors'][] = "Row " . ($index + 2) . ": Order '{$orderRef}' not found in the system";
                            $results['skipped']++;
                            continue;
                        }
                        
                        // Check if already marked as returned
                        if ($order->returned_at) {
                            $results['errors'][] = "Row " . ($index + 2) . ": Order '{$orderRef}' was already marked as returned on " . $order->returned_at->format('Y-m-d');
                            $results['skipped']++;
                            continue;
                        }
                        
                        // Mark order as returned
                        $order->update([
                            'status' => 'returned',
                            'returned_at' => now()
                        ]);
                        
                        $results['success']++;
                        $results['orders_marked_returned']++;
                        
                        // Collect returned products info
                        foreach ($order->orderItems as $orderItem) {
                            $productName = $orderItem->product ? $orderItem->product->name : 'Unknown Product';
                            $productSku = $orderItem->product ? $orderItem->product->sku : 'N/A';
                            
                            // Track returned quantities per product
                            $productKey = $orderItem->product_id;
                            if (!isset($results['returned_products'][$productKey])) {
                                $results['returned_products'][$productKey] = [
                                    'product_id' => $orderItem->product_id,
                                    'product_name' => $productName,
                                    'sku' => $productSku,
                                    'total_quantity' => 0,
                                    'orders' => []
                                ];
                            }
                            
                            $results['returned_products'][$productKey]['total_quantity'] += $orderItem->quantity;
                            $results['returned_products'][$productKey]['orders'][] = [
                                'order_ref' => $orderRef,
                                'quantity' => $orderItem->quantity
                            ];
                        }
                        
                        // Restore inventory if enabled
                        if ($restoreInventory && $order->orderItems) {
                            $this->handleInventoryRestoration($order, $results);
                        }
                        
                    } catch (\Exception $e) {
                        $results['errors'][] = "Row " . ($index + 2) . ": " . $e->getMessage();
                        $results['skipped']++;
                    }
                }
            });
            
        } catch (\Exception $e) {
            $results['errors'][] = "File processing error: " . $e->getMessage();
        }
        
        return $results;
    }
    
    /**
     * Validate that required columns exist in the returns Excel file
     */
    private function validateReturnsHeaders(array $headers): void
    {
        $requiredColumns = ['ORDER_REFERENCE_NUMBER'];
        
        foreach ($requiredColumns as $required) {
            $found = false;
            foreach ($headers as $header) {
                if (strcasecmp(trim($header), $required) === 0) {
                    $found = true;
                    break;
                }
            }
            
            if (!$found) {
                throw new \Exception("Required column '{$required}' not found in Excel file. Please ensure your file has the correct format.");
            }
        }
    }
    
    /**
     * Handle inventory restoration for returned orders
     */
    private function handleInventoryRestoration(Order $order, array &$results): void
    {
        foreach ($order->orderItems as $orderItem) {
            if (!$orderItem->product) {
                continue;
            }
            
            try {
                // Restore raw materials for each product in the order
                $product = $orderItem->product;
                $quantity = $orderItem->quantity;
                
                // Add back raw materials based on the product's recipe
                foreach ($product->rawMaterials as $rawMaterial) {
                    $quantityToRestore = $rawMaterial->pivot->quantity_required * $quantity;
                    $this->inventoryService->addRawMaterialStock($rawMaterial->id, $quantityToRestore);
                }
                
                $results['inventory_restored'][] = [
                    'product_name' => $product->name,
                    'quantity' => $quantity,
                    'order_ref' => $order->shopify_order_id
                ];
                
            } catch (\Exception $e) {
                $results['errors'][] = "Failed to restore inventory for product '{$orderItem->product->name}' from order '{$order->shopify_order_id}': " . $e->getMessage();
            }
        }
    }
}

