<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Order;
use App\Models\OrderItem;
use App\Models\Product;
use App\Models\RawMaterial;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;

class ReportsController extends Controller
{
    /**
     * Display the reports dashboard with P&L and metrics
     */
    public function index(Request $request)
    {
        // Get filter parameters
        $filterType = $request->get('filter_type', 'monthly');
        $startDate = $request->get('start_date');
        $endDate = $request->get('end_date');
        
        // Calculate date range based on filter type
        $dateRange = $this->calculateDateRange($filterType, $startDate, $endDate);
        
        // Get all metrics
        $metrics = $this->calculateMetrics($dateRange['start'], $dateRange['end']);
        
        // Get product performance data
        $productPerformance = $this->getProductPerformance($dateRange['start'], $dateRange['end']);
        
        // Get returns analysis
        $returnsAnalysis = $this->getReturnsAnalysis($dateRange['start'], $dateRange['end']);
        
        // Get daily/period breakdown
        $periodBreakdown = $this->getPeriodBreakdown($dateRange['start'], $dateRange['end'], $filterType);
        
        return view('reports.index', compact(
            'metrics',
            'productPerformance',
            'returnsAnalysis',
            'periodBreakdown',
            'filterType',
            'startDate',
            'endDate',
            'dateRange'
        ));
    }
    
    /**
     * Calculate date range based on filter type
     */
    private function calculateDateRange($filterType, $startDate = null, $endDate = null)
    {
        $now = Carbon::now();
        
        switch ($filterType) {
            case 'daily':
                return [
                    'start' => $now->copy()->startOfDay(),
                    'end' => $now->copy()->endOfDay(),
                    'label' => 'Today - ' . $now->format('M d, Y')
                ];
                
            case 'weekly':
                return [
                    'start' => $now->copy()->startOfWeek(),
                    'end' => $now->copy()->endOfWeek(),
                    'label' => 'This Week - ' . $now->copy()->startOfWeek()->format('M d') . ' to ' . $now->copy()->endOfWeek()->format('M d, Y')
                ];
                
            case 'monthly':
                return [
                    'start' => $now->copy()->startOfMonth(),
                    'end' => $now->copy()->endOfMonth(),
                    'label' => $now->format('F Y')
                ];
                
            case 'yearly':
                return [
                    'start' => $now->copy()->startOfYear(),
                    'end' => $now->copy()->endOfYear(),
                    'label' => $now->format('Y')
                ];
                
            case 'custom':
                $start = $startDate ? Carbon::parse($startDate)->startOfDay() : $now->copy()->startOfMonth();
                $end = $endDate ? Carbon::parse($endDate)->endOfDay() : $now->copy()->endOfDay();
                return [
                    'start' => $start,
                    'end' => $end,
                    'label' => $start->format('M d, Y') . ' - ' . $end->format('M d, Y')
                ];
                
            default:
                return [
                    'start' => $now->copy()->startOfMonth(),
                    'end' => $now->copy()->endOfMonth(),
                    'label' => $now->format('F Y')
                ];
        }
    }
    
    /**
     * Calculate all P&L and business metrics
     */
    private function calculateMetrics($startDate, $endDate)
    {
        // Active orders (excluding returns)
        $activeOrders = Order::whereNull('returned_at')
            ->whereBetween('order_date', [$startDate, $endDate]);
        
        // Returned orders
        $returnedOrders = Order::whereNotNull('returned_at')
            ->whereBetween('returned_at', [$startDate, $endDate]);
        
        // Revenue metrics
        $grossSales = (clone $activeOrders)->sum('total_amount');
        $shippingCharges = (clone $activeOrders)->sum('shipping_charges');
        $gst = (clone $activeOrders)->sum('gst');
        $whIncTax = (clone $activeOrders)->sum('wh_inc_tax');
        $whSalesTax = (clone $activeOrders)->sum('wh_sales_tax');
        $netRevenue = (clone $activeOrders)->sum('net_amount');
        
        // If net_amount is not set for all orders, calculate it
        // Net = Gross - All Deductions (shipping is a deduction, not income)
        if ($netRevenue == 0 && $grossSales > 0) {
            $netRevenue = $grossSales - $shippingCharges - $gst - $whIncTax - $whSalesTax;
        }
        
        // Returns metrics
        $returnedAmount = (clone $returnedOrders)->sum('total_amount');
        $returnedCount = (clone $returnedOrders)->count();
        
        // Order counts
        $totalOrders = (clone $activeOrders)->count();
        $totalItems = OrderItem::whereHas('order', function($q) use ($startDate, $endDate) {
            $q->whereNull('returned_at')
              ->whereBetween('order_date', [$startDate, $endDate]);
        })->sum('quantity');
        
        // Average order value
        $avgOrderValue = $totalOrders > 0 ? $grossSales / $totalOrders : 0;
        
        // Return rate
        $totalOrdersIncludingReturns = $totalOrders + $returnedCount;
        $returnRate = $totalOrdersIncludingReturns > 0 ? ($returnedCount / $totalOrdersIncludingReturns) * 100 : 0;
        
        return [
            'gross_sales' => $grossSales,
            'shipping_charges' => $shippingCharges,
            'gst' => $gst,
            'wh_inc_tax' => $whIncTax,
            'wh_sales_tax' => $whSalesTax,
            'total_deductions' => $shippingCharges + $gst + $whIncTax + $whSalesTax,
            'net_revenue' => $netRevenue ?: ($grossSales - $shippingCharges - $gst - $whIncTax - $whSalesTax),
            'returned_amount' => $returnedAmount,
            'returned_count' => $returnedCount,
            'total_orders' => $totalOrders,
            'total_items' => $totalItems,
            'avg_order_value' => $avgOrderValue,
            'return_rate' => $returnRate,
        ];
    }
    
    /**
     * Get product performance data
     */
    private function getProductPerformance($startDate, $endDate)
    {
        return OrderItem::select(
                'product_id',
                DB::raw('SUM(quantity) as total_quantity'),
                DB::raw('SUM(total_price) as total_revenue'),
                DB::raw('COUNT(DISTINCT order_id) as order_count')
            )
            ->whereHas('order', function($q) use ($startDate, $endDate) {
                $q->whereNull('returned_at')
                  ->whereBetween('order_date', [$startDate, $endDate]);
            })
            ->with('product')
            ->groupBy('product_id')
            ->orderByDesc('total_revenue')
            ->limit(10)
            ->get();
    }
    
    /**
     * Get returns analysis
     */
    private function getReturnsAnalysis($startDate, $endDate)
    {
        $returnedOrders = Order::whereNotNull('returned_at')
            ->whereBetween('returned_at', [$startDate, $endDate])
            ->with('orderItems.product')
            ->get();
        
        $returnedProducts = [];
        
        foreach ($returnedOrders as $order) {
            foreach ($order->orderItems as $orderItem) {
                $productId = $orderItem->product_id;
                
                if (!isset($returnedProducts[$productId])) {
                    $returnedProducts[$productId] = [
                        'product' => $orderItem->product,
                        'total_quantity' => 0,
                        'total_value' => 0,
                        'return_count' => 0
                    ];
                }
                
                $returnedProducts[$productId]['total_quantity'] += $orderItem->quantity;
                $returnedProducts[$productId]['total_value'] += $orderItem->total_price;
                $returnedProducts[$productId]['return_count']++;
            }
        }
        
        // Sort by quantity
        usort($returnedProducts, function($a, $b) {
            return $b['total_quantity'] - $a['total_quantity'];
        });
        
        return array_slice($returnedProducts, 0, 10);
    }
    
    /**
     * Get period breakdown (daily/weekly data points)
     */
    private function getPeriodBreakdown($startDate, $endDate, $filterType)
    {
        $breakdown = [];
        
        if ($filterType == 'yearly') {
            // Monthly breakdown for yearly view
            $current = $startDate->copy()->startOfMonth();
            
            while ($current <= $endDate) {
                $monthEnd = $current->copy()->endOfMonth();
                if ($monthEnd > $endDate) {
                    $monthEnd = $endDate->copy();
                }
                
                $revenue = Order::whereNull('returned_at')
                    ->whereBetween('order_date', [$current, $monthEnd])
                    ->sum('total_amount');
                
                $orders = Order::whereNull('returned_at')
                    ->whereBetween('order_date', [$current, $monthEnd])
                    ->count();
                
                $breakdown[] = [
                    'label' => $current->format('M Y'),
                    'revenue' => $revenue,
                    'orders' => $orders
                ];
                
                $current->addMonth();
            }
        } elseif ($filterType == 'monthly' || $filterType == 'custom') {
            // Daily breakdown for monthly/custom view
            $current = $startDate->copy();
            
            while ($current <= $endDate) {
                $dayEnd = $current->copy()->endOfDay();
                
                $revenue = Order::whereNull('returned_at')
                    ->whereBetween('order_date', [$current, $dayEnd])
                    ->sum('total_amount');
                
                $orders = Order::whereNull('returned_at')
                    ->whereBetween('order_date', [$current, $dayEnd])
                    ->count();
                
                $breakdown[] = [
                    'label' => $current->format('M d'),
                    'revenue' => $revenue,
                    'orders' => $orders
                ];
                
                $current->addDay();
            }
        }
        
        return $breakdown;
    }
    
    /**
     * Simple table-based report view
     */
    public function simple(Request $request)
    {
        // Get filter parameters
        $filterType = $request->get('filter_type', 'monthly');
        $startDate = $request->get('start_date');
        $endDate = $request->get('end_date');
        
        // Calculate date range based on filter type
        $dateRange = $this->calculateDateRange($filterType, $startDate, $endDate);
        
        // Get orders with items
        $orders = Order::with('orderItems.product')
            ->whereNull('returned_at')
            ->whereBetween('order_date', [$dateRange['start'], $dateRange['end']])
            ->orderBy('order_date', 'desc')
            ->get();
        
        // Calculate totals
        $totals = [
            'orders_count' => $orders->count(),
            'gross_sales' => $orders->sum('total_amount'),
            'shipping' => $orders->sum('shipping_charges'),
            'gst' => $orders->sum('gst'),
            'wh_inc_tax' => $orders->sum('wh_inc_tax'),
            'wh_sales_tax' => $orders->sum('wh_sales_tax'),
            'net_revenue' => $orders->sum('net_amount'),
            'total_items' => $orders->sum(function($order) {
                return $order->orderItems->sum('quantity');
            })
        ];
        
        // If net_amount is not set, calculate it
        // Net = Gross - All Deductions (shipping is a courier deduction, not income)
        if ($totals['net_revenue'] == 0 && $totals['gross_sales'] > 0) {
            $totals['net_revenue'] = $totals['gross_sales'] - $totals['shipping'] - $totals['gst'] - $totals['wh_inc_tax'] - $totals['wh_sales_tax'];
        }
        
        return view('reports.simple', compact(
            'orders',
            'totals',
            'filterType',
            'startDate',
            'endDate',
            'dateRange'
        ));
    }
    
    /**
     * Export report data (could be CSV, PDF, etc.)
     */
    public function export(Request $request)
    {
        // Future implementation for exporting reports
        return response()->json(['message' => 'Export functionality coming soon']);
    }
}
