<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Order;
use App\Services\InventoryService;

class OrderController extends Controller
{
    protected $inventoryService;

    public function __construct(InventoryService $inventoryService)
    {
        $this->inventoryService = $inventoryService;
    }

    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $orders = Order::with('orderItems.product')->paginate(15);
        return view('orders.index', compact('orders'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return view('orders.create');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $request->validate([
            'shopify_order_id' => 'required|string|unique:orders',
            'customer_name' => 'required|string|max:255',
            'customer_email' => 'nullable|email',
            'total_amount' => 'required|numeric|min:0',
            'order_date' => 'required|date',
            'status' => 'required|string|in:pending,processing,shipped,delivered,cancelled',
            'notes' => 'nullable|string'
        ]);

        Order::create($request->all());

        return redirect()->route('orders.index')->with('success', 'Order created successfully');
    }

    /**
     * Display the specified resource.
     */
    public function show(Order $order)
    {
        $order->load('orderItems.product');
        $shortages = $this->inventoryService->checkStockAvailability($order);
        return view('orders.show', compact('order', 'shortages'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Order $order)
    {
        return view('orders.edit', compact('order'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Order $order)
    {
        $request->validate([
            'shopify_order_id' => 'required|string|unique:orders,shopify_order_id,' . $order->id,
            'customer_name' => 'required|string|max:255',
            'customer_email' => 'nullable|email',
            'total_amount' => 'required|numeric|min:0',
            'order_date' => 'required|date',
            'status' => 'required|string|in:pending,processing,shipped,delivered,cancelled',
            'notes' => 'nullable|string'
        ]);

        $order->update($request->all());

        return redirect()->route('orders.index')->with('success', 'Order updated successfully');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Order $order)
    {
        $order->delete();
        return redirect()->route('orders.index')->with('success', 'Order deleted successfully');
    }
    
    /**
     * Display returned products summary
     */
    public function returns()
    {
        // Get all returned orders with their items
        $returnedOrders = Order::whereNotNull('returned_at')
            ->with('orderItems.product')
            ->orderBy('returned_at', 'desc')
            ->get();
        
        // Group products by ID and sum quantities
        $returnedProducts = [];
        
        foreach ($returnedOrders as $order) {
            foreach ($order->orderItems as $orderItem) {
                $productId = $orderItem->product_id;
                
                if (!isset($returnedProducts[$productId])) {
                    $returnedProducts[$productId] = [
                        'product' => $orderItem->product,
                        'total_quantity' => 0,
                        'orders' => []
                    ];
                }
                
                $returnedProducts[$productId]['total_quantity'] += $orderItem->quantity;
                $returnedProducts[$productId]['orders'][] = [
                    'order' => $order,
                    'quantity' => $orderItem->quantity,
                    'unit_price' => $orderItem->unit_price,
                    'total_price' => $orderItem->total_price
                ];
            }
        }
        
        return view('orders.returns', compact('returnedProducts', 'returnedOrders'));
    }
}
