<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Services\InventoryService;
use App\Models\Product;
use App\Models\RawMaterial;
use App\Models\Order;

class InventoryController extends Controller
{
    protected $inventoryService;

    public function __construct(InventoryService $inventoryService)
    {
        $this->inventoryService = $inventoryService;
    }

    /**
     * Display the inventory dashboard
     */
    public function dashboard()
    {
        $inventoryStatus = $this->inventoryService->getInventoryStatus();
        
        // Get products with their calculated "can make" values
        $products = Product::where('is_active', true)
            ->with('rawMaterials')
            ->get()
            ->map(function($product) {
                return [
                    'id' => $product->id,
                    'name' => $product->name,
                    'sku' => $product->sku,
                    'can_make' => $product->calculateCanMake()
                ];
            })
            ->toArray();
            
        $recentOrders = Order::with('orderItems.product')
            ->orderBy('created_at', 'desc')
            ->limit(10)
            ->get();

        return view('inventory.dashboard', compact('inventoryStatus', 'recentOrders') + ['lowStockProducts' => $products]);
    }

    /**
     * Display raw material inventory status
     */
    public function rawMaterials()
    {
        $rawMaterials = RawMaterial::where('is_active', true)
            ->orderBy('name')
            ->get();

        return view('inventory.raw-materials', compact('rawMaterials'));
    }

    /**
     * Display product inventory status
     */
    public function products()
    {
        $products = Product::where('is_active', true)
            ->with('rawMaterials')
            ->orderBy('name')
            ->get();

        return view('inventory.products', compact('products'));
    }

    /**
     * Add stock to raw material
     */
    public function addRawMaterialStock(Request $request)
    {
        $request->validate([
            'raw_material_id' => 'required|exists:raw_materials,id',
            'quantity' => 'required|numeric|min:0.01'
        ]);

        $this->inventoryService->addRawMaterialStock(
            $request->raw_material_id,
            $request->quantity
        );

        return redirect()->back()->with('success', 'Stock added successfully');
    }

    /**
     * Check stock availability for an order
     */
    public function checkOrderStock(Order $order)
    {
        $shortages = $this->inventoryService->checkStockAvailability($order);
        
        return response()->json([
            'has_shortages' => !empty($shortages),
            'shortages' => $shortages
        ]);
    }

    /**
     * Process order and update inventory (Manual process - only needed if import was done without inventory deduction)
     */
    public function processOrder(Order $order)
    {
        // Check if inventory was already deducted (order status indicates this)
        if ($order->status === 'processing' || $order->status === 'shipped') {
            return redirect()->back()->with('warning', 'This order has already been processed. Inventory was likely deducted during import.');
        }
        
        $shortages = $this->inventoryService->checkStockAvailability($order);
        
        if (!empty($shortages)) {
            $shortageList = implode(', ', array_map(function($s) {
                return "{$s['raw_material']->name} (need {$s['required']}, have {$s['available']})";
            }, $shortages));
            
            return redirect()->back()->with('error', "Insufficient raw material stock: {$shortageList}");
        }

        $this->inventoryService->updateRawMaterialStock($order);
        $order->update(['status' => 'processing']);

        return redirect()->back()->with('success', 'Order processed and inventory updated manually');
    }
}
