<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Services\ImportService;

class ImportController extends Controller
{
    protected $importService;

    public function __construct(ImportService $importService)
    {
        $this->importService = $importService;
    }

    /**
     * Show the import form
     */
    public function index()
    {
        return view('import.index');
    }

    /**
     * Handle Excel file upload and import
     */
    public function import(Request $request)
    {
        $request->validate([
            'file' => 'required|file|mimes:xlsx,xls,csv|max:10240', // 10MB max
            'deduct_inventory' => 'nullable|boolean'
        ]);

        try {
            $deductInventory = $request->input('deduct_inventory', true);
            $results = $this->importService->importOrdersFromExcel(
                $request->file('file'), 
                $deductInventory
            );
            
            $message = "✓ Order import completed!\n\n";
            $message .= "📦 Orders Created: {$results['orders_created']}\n";
            $message .= "📋 Line Items: {$results['line_items_created']}\n";
            $message .= "⚠️ Skipped: {$results['skipped']}\n";
            
            if (!empty($results['inventory_warnings'])) {
                $message .= "\n⚠️ Inventory Warnings:\n";
                $message .= implode("\n", $results['inventory_warnings']);
            }
            
            if (!empty($results['errors'])) {
                $message .= "\n\n❌ Errors:\n" . implode("\n", $results['errors']);
            }
            
            $messageType = empty($results['errors']) ? 'success' : 'warning';
            
            return redirect()->back()
                ->with('import_results', $results)
                ->with($messageType, $message);
            
        } catch (\Exception $e) {
            return redirect()->back()->with('error', 'Import failed: ' . $e->getMessage());
        }
    }

    /**
     * Handle product Excel file upload and import
     */
    public function importProducts(Request $request)
    {
        $request->validate([
            'file' => 'required|file|mimes:xlsx,xls,csv|max:10240' // 10MB max
        ]);

        try {
            $results = $this->importService->importProductsFromExcel($request->file('file'));
            
            $message = "Product import completed. ";
            $message .= "Success: {$results['success']}, ";
            $message .= "Skipped: {$results['skipped']}, ";
            $message .= "Errors: " . count($results['errors']);
            
            if (!empty($results['errors'])) {
                $message .= "\n\nErrors:\n" . implode("\n", $results['errors']);
            }
            
            return redirect()->back()->with('import_results', $results)->with('message', $message);
            
        } catch (\Exception $e) {
            return redirect()->back()->with('error', 'Product import failed: ' . $e->getMessage());
        }
    }

    /**
     * Download sample Excel template for orders (Shopify format)
     */
    public function downloadTemplate()
    {
        $spreadsheet = new \PhpOffice\PhpSpreadsheet\Spreadsheet();
        $sheet = $spreadsheet->getActiveSheet();
        
        // Set headers matching Shopify export format
        $headers = [
            'Name', 
            'Email', 
            'Billing Name', 
            'Financial Status', 
            'Fulfillment Status',
            'Lineitem name',
            'Lineitem quantity',
            'Lineitem price',
            'Lineitem sku',
            'Total'
        ];
        $sheet->fromArray($headers, NULL, 'A1');
        
        // Add sample data showing how multi-line items work
        // Note: Common order data only on first row, subsequent rows have line item data only
        $sampleData = [
            // Order 1 - 2 line items
            ['#1001', 'john@example.com', 'John Doe', 'paid', 'fulfilled', 'Product A', '2', '25.00', 'SKU-001', '100.00'],
            ['', '', '', '', '', 'Product B', '1', '50.00', 'SKU-002', ''], // Same order, only line item data
            
            // Order 2 - 1 line item
            ['#1002', 'jane@example.com', 'Jane Smith', 'paid', 'unfulfilled', 'Product C', '3', '15.00', 'SKU-003', '45.00'],
            
            // Order 3 - 3 line items
            ['#1003', 'bob@example.com', 'Bob Johnson', 'pending', 'unfulfilled', 'Product A', '1', '25.00', 'SKU-001', '90.00'],
            ['', '', '', '', '', 'Product D', '2', '20.00', 'SKU-004', ''],
            ['', '', '', '', '', 'Product E', '1', '25.00', 'SKU-005', '']
        ];
        $sheet->fromArray($sampleData, NULL, 'A2');
        
        // Style the header row
        $headerStyle = [
            'font' => ['bold' => true],
            'fill' => ['fillType' => \PhpOffice\PhpSpreadsheet\Style\Fill::FILL_SOLID, 'startColor' => ['rgb' => 'E2E8F0']]
        ];
        $sheet->getStyle('A1:J1')->applyFromArray($headerStyle);
        
        // Add a note explaining the format
        $sheet->setCellValue('A10', 'Note: For orders with multiple line items, common order data (Name, Email, etc.) only appears in the first row.');
        $sheet->getStyle('A10')->getFont()->setItalic(true)->setSize(10);
        $sheet->mergeCells('A10:J10');
        
        // Auto-size columns
        foreach(range('A','J') as $col) {
            $sheet->getColumnDimension($col)->setAutoSize(true);
        }
        
        $writer = new \PhpOffice\PhpSpreadsheet\Writer\Xlsx($spreadsheet);
        
        $fileName = 'shopify_order_import_template.xlsx';
        $temp_file = tempnam(sys_get_temp_dir(), $fileName);
        $writer->save($temp_file);
        
        return response()->download($temp_file, $fileName)->deleteFileAfterSend(true);
    }
    
    /**
     * Download sample Excel template for products
     */
    public function downloadProductTemplate()
    {
        $spreadsheet = new \PhpOffice\PhpSpreadsheet\Spreadsheet();
        $sheet = $spreadsheet->getActiveSheet();
        
        // Set headers
        $headers = ['Name', 'Price'];
        $sheet->fromArray($headers, NULL, 'A1');
        
        // Add sample data
        $sampleData = [
            ['Product Name 1', '29.99'],
            ['Product Name 2', '49.99'],
            ['Product Name 3', '19.99']
        ];
        $sheet->fromArray($sampleData, NULL, 'A2');
        
        // Style the header row
        $headerStyle = [
            'font' => ['bold' => true],
            'fill' => ['fillType' => \PhpOffice\PhpSpreadsheet\Style\Fill::FILL_SOLID, 'startColor' => ['rgb' => 'E2E8F0']]
        ];
        $sheet->getStyle('A1:B1')->applyFromArray($headerStyle);
        
        // Auto-size columns
        $sheet->getColumnDimension('A')->setAutoSize(true);
        $sheet->getColumnDimension('B')->setAutoSize(true);
        
        $writer = new \PhpOffice\PhpSpreadsheet\Writer\Xlsx($spreadsheet);
        
        $fileName = 'product_import_template.xlsx';
        $temp_file = tempnam(sys_get_temp_dir(), $fileName);
        $writer->save($temp_file);
        
        return response()->download($temp_file, $fileName)->deleteFileAfterSend(true);
    }
    
    /**
     * Handle courier data Excel file upload and import
     */
    public function importCourier(Request $request)
    {
        $request->validate([
            'file' => 'required|file|mimes:xlsx,xls,csv|max:10240' // 10MB max
        ]);

        try {
            $results = $this->importService->importCourierDataFromExcel($request->file('file'));
            
            $message = "✓ Courier data import completed!\n\n";
            $message .= "📦 Orders Updated: {$results['success']}\n";
            $message .= "⚠️ Skipped: {$results['skipped']}\n";
            
            if (!empty($results['updated_orders'])) {
                $message .= "\n✅ Updated Orders Summary:\n";
                foreach ($results['updated_orders'] as $order) {
                    $message .= sprintf(
                        "• %s: Original: Rs.%.2f + Shipping: Rs.%.2f = Rs.%.2f | Deductions: Rs.%.2f | Net: Rs.%.2f\n",
                        $order['order_ref'],
                        $order['original_amount'],
                        $order['shipping_charges'],
                        $order['total_with_shipping'],
                        $order['deductions'],
                        $order['net_amount']
                    );
                }
            }
            
            if (!empty($results['errors'])) {
                $message .= "\n\n❌ Errors:\n" . implode("\n", $results['errors']);
            }
            
            $messageType = empty($results['errors']) ? 'success' : 'warning';
            
            return redirect()->back()
                ->with('import_results', $results)
                ->with($messageType, $message);
            
        } catch (\Exception $e) {
            return redirect()->back()->with('error', 'Courier data import failed: ' . $e->getMessage());
        }
    }
    
    /**
     * Download sample Excel template for courier data
     */
    public function downloadCourierTemplate()
    {
        $spreadsheet = new \PhpOffice\PhpSpreadsheet\Spreadsheet();
        $sheet = $spreadsheet->getActiveSheet();
        
        // Set headers
        $headers = [
            'Order Ref.',
            'ShippingCharges',
            'GST',
            'WH Inc.Tax (2%)',
            'WH SalesTax (2%)',
            'NetAmount'
        ];
        $sheet->fromArray($headers, NULL, 'A1');
        
        // Add sample data with explanations
        $sampleData = [
            ['#1001', '150.00', '200.00', '45.00', '30.00', '2875.00'],
            ['#1002', '120.00', '180.00', '38.00', '25.00', '2377.00'],
            ['#1003', '100.00', '150.00', '32.00', '22.00', '1996.00']
        ];
        $sheet->fromArray($sampleData, NULL, 'A2');
        
        // Style the header row
        $headerStyle = [
            'font' => ['bold' => true],
            'fill' => ['fillType' => \PhpOffice\PhpSpreadsheet\Style\Fill::FILL_SOLID, 'startColor' => ['rgb' => 'E2E8F0']]
        ];
        $sheet->getStyle('A1:F1')->applyFromArray($headerStyle);
        
        // Add notes
        $sheet->setCellValue('A6', 'Column Descriptions:');
        $sheet->getStyle('A6')->getFont()->setBold(true);
        
        $notes = [
            'Order Ref.: Must match the Order ID in the system (e.g., #1001)',
            'ShippingCharges: Delivery charges from courier service',
            'GST: Goods and Services Tax deducted',
            'WH Inc.Tax (2%): Withholding Income Tax (percentage may vary)',
            'WH SalesTax (2%): Withholding Sales Tax (percentage may vary)',
            'NetAmount: Final amount you will receive from courier service',
            'Note: Column names must include percentage in parentheses, e.g., "WH Inc.Tax (2%)"'
        ];
        
        $row = 7;
        foreach ($notes as $note) {
            $sheet->setCellValue('A' . $row, $note);
            $sheet->getStyle('A' . $row)->getFont()->setSize(9)->setItalic(true);
            $row++;
        }
        
        // Auto-size columns
        foreach(range('A','F') as $col) {
            $sheet->getColumnDimension($col)->setAutoSize(true);
        }
        
        $writer = new \PhpOffice\PhpSpreadsheet\Writer\Xlsx($spreadsheet);
        
        $fileName = 'courier_data_import_template.xlsx';
        $temp_file = tempnam(sys_get_temp_dir(), $fileName);
        $writer->save($temp_file);
        
        return response()->download($temp_file, $fileName)->deleteFileAfterSend(true);
    }
    
    /**
     * Handle returns Excel file upload and import
     */
    public function importReturns(Request $request)
    {
        $request->validate([
            'file' => 'required|file|mimes:xlsx,xls,csv|max:10240', // 10MB max
            'restore_inventory' => 'nullable|boolean'
        ]);

        try {
            $restoreInventory = $request->input('restore_inventory', true);
            $results = $this->importService->importReturnsFromExcel(
                $request->file('file'),
                $restoreInventory
            );
            
            $message = "✓ Returns import completed!\n\n";
            $message .= "🔄 Orders Marked as Returned: {$results['orders_marked_returned']}\n";
            $message .= "⚠️ Skipped: {$results['skipped']}\n";
            
            // Show returned products summary
            if (!empty($results['returned_products'])) {
                $message .= "\n📦 Returned Products Summary:\n";
                foreach ($results['returned_products'] as $product) {
                    $message .= sprintf(
                        "• %s (%s): %d units returned\n",
                        $product['product_name'],
                        $product['sku'],
                        $product['total_quantity']
                    );
                }
            }
            
            // Show inventory restoration
            if (!empty($results['inventory_restored'])) {
                $message .= "\n✅ Inventory Restored:\n";
                foreach ($results['inventory_restored'] as $restored) {
                    $message .= sprintf(
                        "• %s: +%d units (from order %s)\n",
                        $restored['product_name'],
                        $restored['quantity'],
                        $restored['order_ref']
                    );
                }
            }
            
            if (!empty($results['errors'])) {
                $message .= "\n\n❌ Errors:\n" . implode("\n", $results['errors']);
            }
            
            $messageType = empty($results['errors']) ? 'success' : 'warning';
            
            return redirect()->back()
                ->with('import_results', $results)
                ->with($messageType, $message);
            
        } catch (\Exception $e) {
            return redirect()->back()->with('error', 'Returns import failed: ' . $e->getMessage());
        }
    }
    
    /**
     * Download sample Excel template for returns
     */
    public function downloadReturnsTemplate()
    {
        $spreadsheet = new \PhpOffice\PhpSpreadsheet\Spreadsheet();
        $sheet = $spreadsheet->getActiveSheet();
        
        // Set headers
        $headers = ['ORDER_REFERENCE_NUMBER'];
        $sheet->fromArray($headers, NULL, 'A1');
        
        // Add sample data
        $sampleData = [
            ['#1001'],
            ['#1005'],
            ['#1012']
        ];
        $sheet->fromArray($sampleData, NULL, 'A2');
        
        // Style the header row
        $headerStyle = [
            'font' => ['bold' => true],
            'fill' => ['fillType' => \PhpOffice\PhpSpreadsheet\Style\Fill::FILL_SOLID, 'startColor' => ['rgb' => 'E2E8F0']]
        ];
        $sheet->getStyle('A1')->applyFromArray($headerStyle);
        
        // Add notes
        $sheet->setCellValue('A6', 'Instructions:');
        $sheet->getStyle('A6')->getFont()->setBold(true);
        
        $notes = [
            '1. Enter one order reference number per row',
            '2. The order reference must match exactly with orders in the system',
            '3. Orders will be marked as "returned" and inventory will be restored',
            '4. Products from returned orders will be tracked for P&L reporting',
            '5. Already returned orders will be skipped with a warning'
        ];
        
        $row = 7;
        foreach ($notes as $note) {
            $sheet->setCellValue('A' . $row, $note);
            $sheet->getStyle('A' . $row)->getFont()->setSize(9)->setItalic(true);
            $row++;
        }
        
        // Auto-size column
        $sheet->getColumnDimension('A')->setAutoSize(true);
        
        $writer = new \PhpOffice\PhpSpreadsheet\Writer\Xlsx($spreadsheet);
        
        $fileName = 'returns_import_template.xlsx';
        $temp_file = tempnam(sys_get_temp_dir(), $fileName);
        $writer->save($temp_file);
        
        return response()->download($temp_file, $fileName)->deleteFileAfterSend(true);
    }
}
