# Order Returns Implementation Guide

## Overview

The system now supports importing and tracking order returns. This feature is designed to help you:
- Mark orders as returned
- Track which products were returned and in what quantities
- Automatically restore inventory when products are returned
- Exclude returned orders from P&L (Profit & Loss) reports

## Features

### 1. Returns Import
- Import returns data from an Excel file with just the order reference number
- Automatically marks orders as "returned" with a timestamp
- Restores raw material inventory based on product recipes
- Tracks all returned products and their quantities

### 2. Returns Tracking
- View all returned orders in a dedicated returns report
- See returned products grouped by item with total quantities
- Track return dates and customer information
- View detailed breakdown of which orders each product was returned from

### 3. P&L Reporting Support
- Returned orders are marked with `returned_at` timestamp
- Can be filtered out from profit calculations
- Maintains complete audit trail of returns

## How to Use

### Step 1: Import Returns

1. Navigate to **Import Data** page (Menu → Import)
2. Click on the **"Import Returns"** tab
3. Prepare your Excel file with the following column:
   - `ORDER_REFERENCE_NUMBER` - The order ID (e.g., #1001)
4. Upload the Excel file
5. Choose whether to restore inventory (recommended: keep checked)
6. Click **"Import Returns"**

### Step 2: Download Template (Optional)

If you need a template:
1. Click **"Download Template"** button on the Import Returns tab
2. Fill in the order reference numbers
3. Upload the completed file

### Step 3: View Returns Report

1. Navigate to **Orders** page
2. Click **"View Returns"** button (yellow button at top)
3. You'll see:
   - Summary cards showing total returned orders, products, and value
   - **Returned Products by Item** - Shows each product with total quantities returned
   - **Returns Timeline** - Shows all returned orders chronologically

### Step 4: Export Returns Data for P&L

When generating P&L reports, you can filter out returned orders by checking:
```php
// In your reports, exclude returned orders
$orders = Order::whereNull('returned_at')->get();
```

## Technical Details

### Database Changes

A new column was added to the `orders` table:
- `returned_at` (timestamp, nullable) - Records when an order was marked as returned

### Order Status

When an order is marked as returned:
- `status` field is set to `"returned"`
- `returned_at` timestamp is recorded
- Order appears with a red "Returned" badge in the orders list

### Inventory Restoration

When inventory restoration is enabled (default):
1. System retrieves all products from the returned order
2. For each product, it looks up the raw materials recipe
3. Calculates the total raw materials that were used
4. Adds those quantities back to the raw materials inventory

**Example:**
- Order #1001 had 2 units of "Product A"
- Product A requires: 100g Flour, 50g Sugar
- System restores: 200g Flour, 100g Sugar

### Error Handling

The system handles several scenarios:
- **Order not found**: Skips with error message
- **Already returned**: Skips with warning showing previous return date
- **Missing products**: Logs error but continues with other orders
- **Inventory restoration failures**: Records error but completes the return marking

## Excel File Format

### Required Column
```
ORDER_REFERENCE_NUMBER
#1001
#1005
#1012
```

That's it! Just one column with order reference numbers.

### Tips for Excel File
- Remove any extra formatting
- Ensure order references match exactly (including # symbol)
- One order per row
- Empty rows are automatically skipped

## Viewing Returns in Orders

### Orders List Page
- Returned orders show a red "Returned" badge instead of status
- Return date is displayed below the badge

### Order Detail Page
- Red alert banner at top showing "This order has been returned"
- Return date displayed in order information table
- Note indicating inventory has been restored

## Returns Report Features

### Summary Cards
1. **Total Returned Orders** - Count of all returned orders
2. **Unique Products** - Number of different products returned
3. **Total Items Returned** - Sum of all quantities across all products
4. **Total Value** - Sum of all returned order values

### Returned Products Table
- Shows each product that was returned
- Total quantity returned across all orders
- Number of separate returns for that product
- Total value of returns
- **Expandable details** - Click "View Details" to see which specific orders

### Returns Timeline
- Chronological list of all returned orders
- Shows order reference, customer, dates, amount
- Click order reference to view full order details

## Integration with Reports

### P&L Reports
To exclude returns from P&L calculations:

```php
// Get orders excluding returns
$activeOrders = Order::whereNull('returned_at')
    ->whereBetween('order_date', [$startDate, $endDate])
    ->get();
```

### Sales Reports
To include only returned orders:

```php
// Get only returned orders
$returnedOrders = Order::whereNotNull('returned_at')
    ->whereBetween('returned_at', [$startDate, $endDate])
    ->get();
```

### Inventory Reports
Returns automatically update inventory:
- Check `inventory_transactions` or raw material current stock
- System maintains history of inventory changes

## Best Practices

1. **Import returns promptly** - Import return data as soon as you receive it
2. **Keep Excel files organized** - Save copies of import files for audit purposes
3. **Review return reports regularly** - Monitor return patterns
4. **Verify inventory restoration** - Check inventory levels after importing returns
5. **Exclude from P&L** - Always filter out returned orders when calculating profits

## Common Issues and Solutions

### Issue: Order Not Found
**Solution**: Verify the order reference number matches exactly. Check if order was imported into system first.

### Issue: Already Returned Warning
**Solution**: This order was previously marked as returned. Check returns report for details.

### Issue: Inventory Not Restored
**Solution**: Check if products have raw material recipes configured. Only products with recipes will restore inventory.

### Issue: Wrong Return Date
**Solution**: Returns use the current date/time when imported. If you need a specific return date, it must be manually updated in the database.

## Future Enhancements

Potential features for future versions:
- Partial returns (only some items from an order)
- Return reasons tracking
- Return status workflow (requested → approved → completed)
- Automatic refund calculations
- Return analytics and trends
- Custom return date in import file

## Support

For issues or questions:
1. Check this documentation
2. Review the import results page for specific error messages
3. Check Laravel logs in `storage/logs/laravel.log`
4. Verify Excel file format matches template

## Related Documentation

- [ORDER_IMPORT_GUIDE.md](ORDER_IMPORT_GUIDE.md) - How to import orders
- [INVENTORY_PROCESSING_EXPLANATION.md](INVENTORY_PROCESSING_EXPLANATION.md) - How inventory works
- [FOCUSED_ON_STOCK_MANAGEMENT.md](FOCUSED_ON_STOCK_MANAGEMENT.md) - Stock management overview

